/*****************************************************************************\
|* Authors: E. B. Chin and N. Sukumar                                        *|
|* Date   : May 2020                                                         *|
|* This code is provided as Supplementary Material to our paper contribution *|
|* in Computer Aided Geometric Design, 2020                                  *|
\*****************************************************************************/
#include <array>
#include <cmath>
#include <iostream>
#include <vector>
/*****************************************************************************\
|* Polyhedron: struct defining a polyhedron in a graph                       *|
|*  - coords: (# of coords) vector holding vertex coordinates as an array    *|
|*  - edges: (# of edges) vector holding index vertex values defining        *|
|*           ordered edge connectivity                                       *|
|*  - faces: (# of faces) vector holding index edge values defining          *|
|*                the face connectivity in counterclockwise orientation      *|
|*  - face_dir: (# of faces) vector defining the direction of edges; +1 for  *|
|*              edges in the correct orientation and -1 for edges which must *|
|*              be reversed                                                  *|
\*****************************************************************************/
struct Polyhedron
{
  std::vector<std::vector<unsigned int>> faces;
  std::vector<std::vector<short>> faces_dir;
  std::vector<std::array<unsigned int, 2>> edges;
  std::vector<std::array<double, 3>> coords;
};
/*****************************************************************************\
|* computeIntegrals: integrates monomials over a polyhedron                  *|
|*  Input:                                                                   *|
|*   - polyhedron: Polyhedron defining the domain of integration             *|
|*   - max_order: all monomials <= max_order will be integrated              *|
|*  Output:                                                                  *|
|*   - integrals: Integral values stored in an Integrals struct              *|
|*  Notes:                                                                   *|
|*   - the closest point to the origin on edges and faces are treated as x_0 *|
\*****************************************************************************/
std::vector<std::vector<double>> computeIntegrals(
  const Polyhedron& polyhedron, size_t max_order)
{
  // Create data structures to store integrals ////////////////////////////////
  std::vector<std::vector<double>> integrals(max_order + 1);
  auto number_of_verts = polyhedron.coords.size();
  auto number_of_edges = polyhedron.edges.size();
  auto number_of_faces = polyhedron.faces.size();
  std::vector<std::vector<std::vector<double>>> edge_ints(max_order + 1);
  std::vector<std::vector<std::vector<double>>> face_ints(max_order + 1);
  for (size_t o{}; o < max_order + 1; ++o)
  {
    auto number_of_integrals = (o + 2) * (o + 1) / 2;
    integrals[o].resize(number_of_integrals);
    edge_ints[o].resize(number_of_integrals);
    face_ints[o].resize(number_of_integrals);
    for (size_t i{}; i < number_of_integrals; ++i)
    {
      edge_ints[o][i].resize(number_of_edges);
      face_ints[o][i].resize(number_of_faces);
    }
  }
  // Edge quantities //////////////////////////////////////////////////////////
  // vector tangent to edge
  std::vector<std::array<double, 3>> edges_t(number_of_edges);
  // signed distance from 1st vert to x0
  std::vector<double> edges_d1(number_of_edges);
  // signed distance from 2nd vert to x0
  std::vector<double> edges_d2(number_of_edges);
  // vector from origin to closest point on edge
  std::vector<std::array<double, 3>> edges_x0(number_of_edges);
  // Edge quantity computations ///////////////////////////////////////////////
  for (size_t e{}; e < number_of_edges; ++e)
  {
    auto edge_t_norm = 0.0;
    for (size_t i{}; i < 3; ++i)
    {
      edges_t[e][i] = polyhedron.coords[polyhedron.edges[e][0]][i]
        - polyhedron.coords[polyhedron.edges[e][1]][i];
      edge_t_norm = edge_t_norm + edges_t[e][i] * edges_t[e][i];
    }
    edge_t_norm = std::sqrt(edge_t_norm);
    for (size_t i{}; i < 3; ++i)
    {
      edges_t[e][i] = edges_t[e][i] / edge_t_norm;
      edges_d1[e] = edges_d1[e]
        + edges_t[e][i] * polyhedron.coords[polyhedron.edges[e][0]][i];
      edges_d2[e] = edges_d2[e]
        - edges_t[e][i] * polyhedron.coords[polyhedron.edges[e][1]][i];
    }
    for (size_t i{}; i < 3; ++i)
    {
      edges_x0[e][i] = polyhedron.coords[polyhedron.edges[e][0]][i]
        - edges_d1[e] * edges_t[e][i];
    }
  }
  // Face quantities //////////////////////////////////////////////////////////
  // signed distance from x0 on face to edge line
  std::vector<std::vector<double>> faces_edges_d(number_of_faces);
  // vector from origin to closest point on face
  std::vector<std::array<double, 3>> faces_x0(number_of_faces);
  // signed distance from origin to face
  std::vector<double> faces_d(number_of_faces);
  // Face quantity computations ///////////////////////////////////////////////
  for (size_t f{}; f < number_of_faces; ++f)
  {
    auto number_of_face_edges = polyhedron.faces[f].size();
    std::vector<std::array<double, 3>> face_edges_t(number_of_face_edges);
    for (size_t e{}; e < number_of_face_edges; ++e)
    {
      auto v_curr = polyhedron.edges[polyhedron.faces[f][e]][0];
      auto v_next = polyhedron.edges[polyhedron.faces[f][e]][1];
      faces_x0[f][0] = faces_x0[f][0] + 
        static_cast<double>(polyhedron.faces_dir[f][e])
        * (polyhedron.coords[v_curr][1] - polyhedron.coords[v_next][1])
        * (polyhedron.coords[v_curr][2] + polyhedron.coords[v_next][2]);
      faces_x0[f][1] = faces_x0[f][1] +
        static_cast<double>(polyhedron.faces_dir[f][e])
        * (polyhedron.coords[v_curr][2] - polyhedron.coords[v_next][2])
        * (polyhedron.coords[v_curr][0] + polyhedron.coords[v_next][0]);
      faces_x0[f][2] = faces_x0[f][2] +
        static_cast<double>(polyhedron.faces_dir[f][e])
        * (polyhedron.coords[v_curr][0] - polyhedron.coords[v_next][0])
        * (polyhedron.coords[v_curr][1] + polyhedron.coords[v_next][1]);
      face_edges_t[e][0] = static_cast<double>(polyhedron.faces_dir[f][e])
        * (polyhedron.coords[v_next][0] - polyhedron.coords[v_curr][0]);
      face_edges_t[e][1] = static_cast<double>(polyhedron.faces_dir[f][e])
        * (polyhedron.coords[v_next][1] - polyhedron.coords[v_curr][1]);
      face_edges_t[e][2] = static_cast<double>(polyhedron.faces_dir[f][e])
        * (polyhedron.coords[v_next][2] - polyhedron.coords[v_curr][2]);
      auto face_edge_t_norm = std::sqrt(face_edges_t[e][0] * face_edges_t[e][0]
        + face_edges_t[e][1] * face_edges_t[e][1]
        + face_edges_t[e][2] * face_edges_t[e][2]);
      face_edges_t[e][0] /= face_edge_t_norm;
      face_edges_t[e][1] /= face_edge_t_norm;
      face_edges_t[e][2] /= face_edge_t_norm;
    }
    auto face_x0_norm = std::sqrt(faces_x0[f][0] * faces_x0[f][0]
      + faces_x0[f][1] * faces_x0[f][1] + faces_x0[f][2] * faces_x0[f][2]);
    faces_x0[f][0] /= face_x0_norm;
    faces_x0[f][1] /= face_x0_norm;
    faces_x0[f][2] /= face_x0_norm;
    std::vector<std::array<double, 3>> face_edges_n(number_of_face_edges);
    for (size_t e{}; e < number_of_face_edges; ++e)
    {
      face_edges_n[e][0] = face_edges_t[e][1] * faces_x0[f][2]
        - face_edges_t[e][2] * faces_x0[f][1];
      face_edges_n[e][1] = face_edges_t[e][2] * faces_x0[f][0]
        - face_edges_t[e][0] * faces_x0[f][2];
      face_edges_n[e][2] = face_edges_t[e][0] * faces_x0[f][1]
        - face_edges_t[e][1] * faces_x0[f][0];
    }
    auto v_0 = polyhedron.edges[polyhedron.faces[f][0]][0];
    faces_d[f] = faces_x0[f][0] * polyhedron.coords[v_0][0]
      + faces_x0[f][1] * polyhedron.coords[v_0][1]
      + faces_x0[f][2] * polyhedron.coords[v_0][2];
    faces_x0[f][0] *= faces_d[f];
    faces_x0[f][1] *= faces_d[f];
    faces_x0[f][2] *= faces_d[f];
    faces_edges_d[f].resize(number_of_face_edges);
    for (size_t e{}; e < number_of_face_edges; ++e)
    {
      auto e_curr = polyhedron.faces[f][e];
      faces_edges_d[f][e] =
        (edges_x0[e_curr][0] - faces_x0[f][0]) * face_edges_n[e][0]
        + (edges_x0[e_curr][1] - faces_x0[f][1]) * face_edges_n[e][1]
        + (edges_x0[e_curr][2] - faces_x0[f][2]) * face_edges_n[e][2];
    }
  }
  // Loop over monomial orders ////////////////////////////////////////////////
  for (int o{}; o < max_order + 1; ++o)
  {
    // Loop over powers of x //////////////////////////////////////////////////
    for (int xp{o}; xp >= 0; --xp)
    {
      // Loop over powers of z ////////////////////////////////////////////////
      for (int zp{}; zp < (o - xp + 1); ++zp)
      {
        auto entry = (o - xp) * (o - xp + 1) / 2 + zp;
        auto yp = o - xp - zp;
        // Compute monomial values at vertices ////////////////////////////////
        std::vector<double> vert_vals(number_of_verts);
        for (size_t v{}; v < polyhedron.coords.size(); ++v)
        {
          vert_vals[v] =
            std::pow(polyhedron.coords[v][0], static_cast<double>(xp)) *
            std::pow(polyhedron.coords[v][1], static_cast<double>(yp)) *
            std::pow(polyhedron.coords[v][2], static_cast<double>(zp));
        }
        // Integrate monomial on edges ////////////////////////////////////////
        for (size_t e{}; e < number_of_edges; ++e)
        {
          edge_ints[o][entry][e] =
            (vert_vals[polyhedron.edges[e][0]] * edges_d1[e]
            + vert_vals[polyhedron.edges[e][1]] * edges_d2[e]) 
            / (static_cast<double>(o) + 1.0);
        }
        if (xp > 0)
        {
          auto prev_entry = entry;
          for (size_t e{}; e < number_of_edges; ++e)
          {
            edge_ints[o][entry][e] += static_cast<double>(xp) * edges_x0[e][0]
              * edge_ints[o - 1][prev_entry][e] 
              / (static_cast<double>(o) + 1.0);
          }
        }
        if (yp > 0)
        {
          auto prev_entry = entry - o + xp;
          for (size_t e{}; e < number_of_edges; ++e)
          {
            edge_ints[o][entry][e] += static_cast<double>(yp) * edges_x0[e][1]
              * edge_ints[o - 1][prev_entry][e]
              / (static_cast<double>(o) + 1.0);
          }
        }
        if (zp > 0)
        {
          auto prev_entry = entry - o + xp - 1;
          for (size_t e{}; e < number_of_edges; ++e)
          {
            edge_ints[o][entry][e] += static_cast<double>(zp) * edges_x0[e][2]
              * edge_ints[o - 1][prev_entry][e]
              / (static_cast<double>(o) + 1.0);
          }
        }
        // Integrate monomial on faces ////////////////////////////////////////
        for (size_t f{}; f < number_of_faces; ++f)
        {
          auto number_of_face_edges = polyhedron.faces[f].size();
          for (size_t e{}; e < number_of_face_edges; ++e)
          {
            auto e_curr = polyhedron.faces[f][e];
            face_ints[o][entry][f] +=
              faces_edges_d[f][e] * edge_ints[o][entry][e_curr]
              / (static_cast<double>(o) + 2.0);
          }
        }
        if (xp > 0)
        {
          auto prev_entry = entry;
          for (size_t f{}; f < number_of_faces; ++f)
          {
            face_ints[o][entry][f] +=
              static_cast<double>(xp) * faces_x0[f][0]
              * face_ints[o - 1][prev_entry][f]
              / (static_cast<double>(o) + 2.0);
          }
        }
        if (yp > 0)
        {
          auto prev_entry = entry - o + xp;
          for (size_t f{}; f < number_of_faces; ++f)
          {
            face_ints[o][entry][f] +=
              static_cast<double>(yp) * faces_x0[f][1]
              * face_ints[o - 1][prev_entry][f]
              / (static_cast<double>(o) + 2.0);
          }
        }
        if (zp > 0)
        {
          auto prev_entry = entry - o + xp - 1;
          for (size_t f{}; f < number_of_faces; ++f)
          {
            face_ints[o][entry][f] +=
              static_cast<double>(zp) * faces_x0[f][2]
              * face_ints[o - 1][prev_entry][f]
              / (static_cast<double>(o) + 2.0);
          }
        }
        // Integrate monomial over polyhedron /////////////////////////////////
        for (size_t f{}; f < number_of_faces; ++f)
        {
          integrals[o][entry] += faces_d[f] * face_ints[o][entry][f]
            / (static_cast<double>(o) + 3.0);
        }
      }
    }
  }
  return integrals;
}
/*****************************************************************************\
|* main: computes monomial integrals over a cube                             *|
\*****************************************************************************/
int main()
{
  // Define a cube in [0, 1]^3 ////////////////////////////////////////////////
  Polyhedron polyhedron;
  polyhedron.coords = {{0.0, 0.0, 0.0},
                       {1.0, 0.0, 0.0},
                       {0.0, 1.0, 0.0},
                       {1.0, 1.0, 0.0},
                       {0.0, 0.0, 1.0},
                       {1.0, 0.0, 1.0},
                       {0.0, 1.0, 1.0},
                       {1.0, 1.0, 1.0}};
  polyhedron.edges = {{0, 1},
                      {2, 3},
                      {0, 2},
                      {1, 3},
                      {4, 5},
                      {6, 7},
                      {4, 6},
                      {5, 7},
                      {0, 4},
                      {2, 6},
                      {1, 5},
                      {3, 7}};
  polyhedron.faces = {{ 2,  1,  3,  0},
                      { 4,  7,  5,  6},
                      { 0, 10,  4,  8},
                      { 9,  5, 11,  1},
                      { 8,  6,  9,  2},
                      { 3, 11,  7, 10}};
  polyhedron.faces_dir = {{ 1,  1, -1, -1},
                          { 1,  1, -1, -1},
                          { 1,  1, -1, -1},
                          { 1,  1, -1, -1},
                          { 1,  1, -1, -1},
                          { 1,  1, -1, -1}};
  // Compute integrals and print results //////////////////////////////////////
  auto integrals = computeIntegrals(polyhedron, 2);
  std::cout << "Volume of the cube = " << integrals[0][0] << std::endl
    << "Integral of x      = " << integrals[1][0] << std::endl
    << "Integral of y      = " << integrals[1][1] << std::endl
    << "Integral of z      = " << integrals[1][2] << std::endl
    << "Integral of x^2    = " << integrals[2][0] << std::endl
    << "Integral of xy     = " << integrals[2][1] << std::endl
    << "Integral of xz     = " << integrals[2][2] << std::endl
    << "Integral of y^2    = " << integrals[2][3] << std::endl
    << "Integral of yz     = " << integrals[2][4] << std::endl
    << "Integral of z^2    = " << integrals[2][5] << std::endl;
  return 0;
}
