/*****************************************************************************\
|* Authors: E. B. Chin and N. Sukumar                                        *|
|* Date   : May 2020                                                         *|
|* This code is provided as Supplementary Material to our paper contribution *|
|* in Computer Aided Geometric Design, 2020                                  *|
\*****************************************************************************/
#include <array>
#include <cmath>
#include <iostream>
#include <vector>
/*****************************************************************************\
|* Polyhedron: struct defining a polyhedron                                  *|
|*  - coords: (# of coords) vector holding vertex coordinates as an array    *|
|*  - face_conns: (# of faces) vector holding index vertex values defining   *|
|*                the face connectivity in counterclockwise orientation      *|
|*  - face_norms: (# of faces) vector holding unit face normals as an array  *|
\*****************************************************************************/
struct Polyhedron
{
  std::vector<std::array<double, 3>> coords;
  std::vector<std::vector<unsigned int>> face_conns;
  std::vector<std::array<double, 3>> face_norms;
};
/*****************************************************************************\
|* Integrals: struct defining integral values                                *|
|*  - I(integ): double storing the value of the integral of (integ)          *|
\*****************************************************************************/
struct Integrals
{
  double I1;
  double Ix, Iy, Iz;
  double Ixx, Iyy, Izz, Iyz, Ixz, Ixy;
};
/*****************************************************************************\
|* computeIntegrals: integrates monomials of degree <= 2 over a polyhedron   *|
|*  Input:                                                                   *|
|*   - polyhedron: Polyhedron defining the domain of integration             *|
|*  Output:                                                                  *|
|*   - integrals: Integral values stored in an Integrals struct              *|
|*  Notes:                                                                   *|
|*   - the first vertex is treated as the origin (for coordinate shift);     *|
|*     therefore, the faces connected to the first vertex are ignored        *|
|*   - the first face vertex is treated as xi_i; therefore, the first and    *|
|*     last edges on the face are ignored                                    *|
\*****************************************************************************/
Integrals computeIntegrals(const Polyhedron& polyhedron)
{
  Integrals integrals{};
  auto number_of_verts = polyhedron.coords.size();
  auto number_of_faces = polyhedron.face_conns.size();
  // Shift to xbar coordinates ////////////////////////////////////////////////
  auto x_ell = polyhedron.coords[0];
  auto xbar = polyhedron.coords;
  for (size_t v{}; v < number_of_verts; ++v)
  {
    xbar[v][0] -= x_ell[0];  xbar[v][1] -= x_ell[1];  xbar[v][2] -= x_ell[2];
  }
  // Loop over faces //////////////////////////////////////////////////////////
  for (size_t f{}; f < number_of_faces; ++f)
  {
    auto number_of_verts_on_face = polyhedron.face_conns[f].size();
    auto skip_face = false;
    for (size_t v{}; v < number_of_verts_on_face; ++v)
    {
      if (polyhedron.face_conns[f][v] == 0)
      {
        skip_face = true;
        break;
      }
    }
    if (~skip_face)
    {
      // serves as origin for the face
      auto xi_i = polyhedron.face_conns[f][0];
      // distance from x_ell to the face
      auto face_d = polyhedron.face_norms[f][0] * xbar[xi_i][0]
        + polyhedron.face_norms[f][1] * xbar[xi_i][1]
        + polyhedron.face_norms[f][2] * xbar[xi_i][2];
      auto number_of_edges_on_face = number_of_verts_on_face;
      // Loop over edges //////////////////////////////////////////////////////
      for (size_t e{1}; e < (number_of_edges_on_face - 1); ++e)
      {
        // first vertex on the edge
        auto v1 = polyhedron.face_conns[f][e];
        // second vertex on the edge
        auto v2 = polyhedron.face_conns[f][e + 1];
        // edge tangent vector
        std::array<double, 3> edge_t{xbar[v2][0] - xbar[v1][0],
          xbar[v2][1] - xbar[v1][1], xbar[v2][2] - xbar[v1][2]};
        // edge length
        auto edge_l = std::sqrt(edge_t[0] * edge_t[0]
          + edge_t[1] * edge_t[1] + edge_t[2] * edge_t[2]);
        edge_t[0] /= edge_l;
        edge_t[1] /= edge_l;
        edge_t[2] /= edge_l;
        // in-plane edge normal
        std::array<double, 3> edge_n{edge_t[1] * polyhedron.face_norms[f][2]
          - edge_t[2] * polyhedron.face_norms[f][1],
          edge_t[2] * polyhedron.face_norms[f][0]
          - edge_t[0] * polyhedron.face_norms[f][2],
          edge_t[0] * polyhedron.face_norms[f][1]
          - edge_t[1] * polyhedron.face_norms[f][0]};
        std::array<double, 3> edge_c{xbar[v1][0] - xbar[xi_i][0],
          xbar[v1][1] - xbar[xi_i][1], xbar[v1][2] - xbar[xi_i][2]};
        // signed distance from xi_i to edge
        auto edge_d = edge_n[0] * edge_c[0] + edge_n[1] * edge_c[1]
          + edge_n[2] * edge_c[2];
        // Add edge contributions to integrals ////////////////////////////////
        integrals.I1 += face_d * edge_d * edge_l;
        integrals.Ix += face_d * edge_d * edge_l *
          (xbar[xi_i][0] + xbar[v1][0] + xbar[v2][0]);
        integrals.Iy += face_d * edge_d * edge_l *
          (xbar[xi_i][1] + xbar[v1][1] + xbar[v2][1]);
        integrals.Iz += face_d * edge_d * edge_l *
          (xbar[xi_i][2] + xbar[v1][2] + xbar[v2][2]);
        integrals.Ixx += face_d * edge_d * edge_l *
          (xbar[v1][0] * xbar[v1][0]
            + xbar[v2][0] * xbar[v2][0]
            + xbar[v1][0] * xbar[v2][0]
            + xbar[xi_i][0]
            * (xbar[v2][0] + xbar[v1][0] + xbar[xi_i][0]));
        integrals.Iyy += face_d * edge_d * edge_l *
          (xbar[v1][1] * xbar[v1][1]
            + xbar[v2][1] * xbar[v2][1]
            + xbar[v1][1] * xbar[v2][1]
            + xbar[xi_i][1]
            * (xbar[v2][1] + xbar[v1][1] + xbar[xi_i][1]));
        integrals.Izz += face_d * edge_d * edge_l *
          (xbar[v1][2] * xbar[v1][2]
            + xbar[v2][2] * xbar[v2][2]
            + xbar[v1][2] * xbar[v2][2]
            + xbar[xi_i][2]
            * (xbar[v2][2] + xbar[v1][2] + xbar[xi_i][2]));
        integrals.Iyz += 0.5 * face_d * edge_d * edge_l *
          (2.0 * xbar[v1][1] * xbar[v1][2]
            + xbar[v1][1] * xbar[v2][2]
            + xbar[v2][1] * xbar[v1][2]
            + 2.0 * xbar[v2][1] * xbar[v2][2]
            + xbar[xi_i][1]
            * (xbar[v2][2] + xbar[v1][2] + xbar[xi_i][2])
            + xbar[xi_i][2]
            * (xbar[v2][1] + xbar[v1][1] + xbar[xi_i][1]));
        integrals.Ixz += 0.5 * face_d * edge_d * edge_l *
          (2.0 * xbar[v1][0] * xbar[v1][2]
            + xbar[v1][0] * xbar[v2][2]
            + xbar[v2][0] * xbar[v1][2]
            + 2.0 * xbar[v2][0] * xbar[v2][2]
            + xbar[xi_i][0]
            * (xbar[v2][2] + xbar[v1][2] + xbar[xi_i][2])
            + xbar[xi_i][2]
            * (xbar[v2][0] + xbar[v1][0] + xbar[xi_i][0]));
        integrals.Ixy += 0.5 * face_d * edge_d * edge_l *
          (2.0 * xbar[v1][0] * xbar[v1][1]
            + xbar[v1][0] * xbar[v2][1]
            + xbar[v2][0] * xbar[v1][1]
            + 2.0 * xbar[v2][0] * xbar[v2][1]
            + xbar[xi_i][0]
            * (xbar[v2][1] + xbar[v1][1] + xbar[xi_i][1])
            + xbar[xi_i][1]
            * (xbar[v2][0] + xbar[v1][0] + xbar[xi_i][0]));
      }
    }
  }
  // Shift to x coordinates ///////////////////////////////////////////////////
  integrals.I1 /= 6.0;
  integrals.Ix /= 24.0;
  integrals.Iy /= 24.0;
  integrals.Iz /= 24.0;
  integrals.Ixx = integrals.Ixx / 60.0
    + x_ell[0] * (2.0 * integrals.Ix + x_ell[0] * integrals.I1);
  integrals.Iyy = integrals.Iyy / 60.0
    + x_ell[1] * (2.0 * integrals.Iy + x_ell[1] * integrals.I1);
  integrals.Izz = integrals.Izz / 60.0
    + x_ell[2] * (2.0 * integrals.Iz + x_ell[2] * integrals.I1);
  integrals.Iyz = integrals.Iyz / 60.0
    + x_ell[1] * integrals.Iz + x_ell[2] * integrals.Iy
    + x_ell[1] * x_ell[2] * integrals.I1;
  integrals.Ixz = integrals.Ixz / 60.0
    + x_ell[0] * integrals.Iz + x_ell[2] * integrals.Ix
    + x_ell[0] * x_ell[2] * integrals.I1;
  integrals.Ixy = integrals.Ixy / 60.0
    + x_ell[0] * integrals.Iy + x_ell[1] * integrals.Ix
    + x_ell[0] * x_ell[1] * integrals.I1;
  integrals.Ix += x_ell[0] * integrals.I1;
  integrals.Iy += x_ell[1] * integrals.I1;
  integrals.Iz += x_ell[2] * integrals.I1;
  return integrals;
}
/*****************************************************************************\
|* main: computes monomial integrals over a cube                             *|
\*****************************************************************************/
int main()
{
  // Define a cube in [0, 1]^3 ////////////////////////////////////////////////
  Polyhedron polyhedron;
  polyhedron.coords = {{0.0, 0.0, 1.0},
                       {1.0, 0.0, 1.0},
                       {1.0, 1.0, 1.0},
                       {0.0, 1.0, 1.0},
                       {0.0, 0.0, 0.0},
                       {1.0, 0.0, 0.0},
                       {1.0, 1.0, 0.0},
                       {0.0, 1.0, 0.0}};
  polyhedron.face_conns = {{0, 1, 2, 3},
                           {5, 4, 7, 6},
                           {4, 0, 3, 7},
                           {4, 5, 1, 0},
                           {1, 5, 6, 2},
                           {3, 2, 6, 7}};
  polyhedron.face_norms = {{ 0.0,  0.0,  1.0},
                           { 0.0,  0.0, -1.0},
                           {-1.0,  0.0,  0.0},
                           { 0.0, -1.0,  0.0},
                           { 1.0,  0.0,  0.0},
                           { 0.0,  1.0,  0.0}};
  // Compute integrals and print results //////////////////////////////////////
  auto integrals = computeIntegrals(polyhedron);
  std::cout << "Volume of the cube = " << integrals.I1 << std::endl
    << "Integral of x      = " << integrals.Ix << std::endl
    << "Integral of y      = " << integrals.Iy << std::endl
    << "Integral of z      = " << integrals.Iz << std::endl
    << "Integral of x^2    = " << integrals.Ixx << std::endl
    << "Integral of xy     = " << integrals.Ixy << std::endl
    << "Integral of xz     = " << integrals.Ixz << std::endl
    << "Integral of y^2    = " << integrals.Iyy << std::endl
    << "Integral of yz     = " << integrals.Iyz << std::endl
    << "Integral of z^2    = " << integrals.Izz << std::endl;
  return 0;
}
