c********************************************************************
c     subroutine laplace2d(n,pointxi,phi,plot)
c     Purpose
c     =======
c     Subroutine to compute the Laplace shape functions phi_J [J=1,n]
c     and its derivatives on a canonical element in the reference 
c     coordinate (xi_1,xi_2) where the element is a polygon with n 
c     nodes (n-gon) that all lie on a unit circle.
c 
c     Author: N. Sukumar, UC Davis
c     Date:   August 2003
c
c     Input
c     =====
c     n          = n-sided polygon (n natural neighbors)
c     pointxi    = array of length two in which the local coordinate
c                  (xi_1, xi_2) is passed
c
c     Output
c     ======
c     phi(.,.)   = 2-d array containing the Laplace
c                  shape function and its derivatives  @(xi_1,xi_2)
c                  phi(1,J) : shape function of node J @(xi_1,xi_2)
c                  phi(2,J) : xi1-derivative of node J @(xi_1,xi_2)
c                  phi(3,J) : xi2-derivative of node J @(xi_1,xi_2)
c********************************************************************
c
      subroutine laplace2d(n,pointxi,phi,plot)
      implicit none
c
      integer n
      real*8 pointxi(2), xi1, xi2
      real*8 coord(2,n), phi(3,n)
      logical plot
      integer nodeminus, nodeplus
      real*8 thickness ! line thickness
c
      logical inside
      external in_polygon
      logical  in_polygon

c     circumcenters      
      real*8 cc(2,n)
c    
c     nodal coordinates, circumcenter and derivatives of the
c     circumcenter for the triangles in the triangulation

      real*8 a1, a2, b1, b2, p1, p1_xi1, p1_xi2, p2, p2_xi1, p2_xi2
      real*8 q1, q1_xi1, q1_xi2, q2, q2_xi1, q2_xi2
c
c     shape function computations
c
      real*8 sI, sI_xi1, sI_xi2
      real*8 hI, hI_xi1, hI_xi2
      real*8 sI_hI, der(2), sum, sumder(2)
c
      integer i
c
      xi1 = pointxi(1)
      xi2 = pointxi(2)
c
      do 10 i = 1,n
         phi(1,i) = 0.d0
         phi(2,i) = 0.d0
         phi(3,i) = 0.d0
10    continue
c
c     nodal coordinates
c
      call polycoord(n,coord)
c
c     check if the point is inside the polygon
c
      inside = in_polygon(n,coord,xi1,xi2)
      if (.not. inside) then
         write(6,*)'Error: Point (',xi1,',',xi2,') NOT in the polygon'
         call errormsg('Aborting . . ')
      endif

      sum = 0.0
      sumder(1) = 0.0
      sumder(2) = 0.0
c
      do 20 i = 1,n
c
c     coordinates of node i
c
         a1 = coord(1,i)
         a2 = coord(2,i)
         if (i .eq. 1) then
            nodeminus = n
            nodeplus  = i+1
         else if (i .eq. n) then
            nodeminus = n-1
            nodeplus  = 1
         else
            nodeminus = i-1
            nodeplus  = i+1
         endif
c
c        triangle [B,A,P]: B(b1,b2), A(a1,a2), P(xi1,xi2)
c
         b1 = coord(1,nodeminus)
         b2 = coord(2,nodeminus)
c
         call circumcenter(b1,b2,a1,a2,xi1,xi2,p1,p2,p1_xi1,p1_xi2,
     &                     p2_xi1,p2_xi2)
c
c        triangle [A,B,P]: A(a1,a2), B(b1,b2), P(xi1,xi2)
c
         b1 = coord(1,nodeplus)
         b2 = coord(2,nodeplus)

         call circumcenter(a1,a2,b1,b2,xi1,xi2,q1,q2,q1_xi1,q1_xi2,
     &                     q2_xi1,q2_xi2)

c        store the circumcenter in array cc
         cc(1,i) = p1
         cc(2,i) = p2

         sI = sqrt((p1 - q1)*(p1 - q1) + (p2 - q2)*(p2 - q2))
         sI_xi1 = ((p1 - q1)*(p1_xi1 - q1_xi1) + 
     &             (p2 - q2)*(p2_xi1 - q2_xi1))/sI
         sI_xi2 = ((p1 - q1)*(p1_xi2 - q1_xi2) + 
     &             (p2 - q2)*(p2_xi2 - q2_xi2))/sI
         hI = sqrt((xi1 - a1)*(xi1 - a1) + (xi2 - a2)*(xi2 - a2))
         hI_xi1 = (xi1 - a1)/hI
         hI_xi2 = (xi2 - a2)/hI

         sI_hI = sI/hI
         der(1) = (sI_xi1 - sI_hI*hI_xi1)/hI
         der(2) = (sI_xi2 - sI_hI*hI_xi2)/hI

         sum = sum + sI_hI
         sumder(1) = sumder(1) + der(1)
         sumder(2) = sumder(2) + der(2)

         phi(1,i) = sI_hI
         phi(2,i) = der(1)
         phi(3,i) = der(2)
20    continue
c
      do 30 i = 1,n
         phi(1,i) = phi(1,i)/sum
         phi(2,i) = (phi(2,i) - phi(1,i)*sumder(1))/sum
         phi(3,i) = (phi(3,i) - phi(1,i)*sumder(2))/sum
30    continue
c
      call consistency(xi1,xi2,n,coord,phi)
c
c     tecplot output
c
      if (plot) then
         call tecplot_nodes(n,coord)
         call tecplot_point(pointxi)
         call tecplot_pointnode(n,pointxi,coord)
         thickness = 0.5
         call tecplot_polygon(n,coord,thickness,"polygon.dat") ! 11 characters
         thickness = 0.3
         call tecplot_polygon(n,cc,thickness,"voronoi.dat") ! 11 characters
      endif
c
      return
      end
c
c********************************************************************
c     subroutine circumcenter(a1,a2,b1,b2,x,y,v1,v2,v1x,v1y,
c    &                        v2x,v2y)
c     Purpose
c     =======
c     Subroutine to determine the circumcenter (v1,v2) and the
c     its derivatives v1,i and v2,i (i = 1,2) for the triangle
c     (p_1,p_2,P), with vertices p_1(a1,a2), p_2(b1,b2) and 
c     P(x,y)
c
c********************************************************************
c
      subroutine circumcenter(a1,a2,b1,b2,x,y,v1,v2,v1x,v1y,
     &                        v2x,v2y)
      implicit none
c
      real*8 TOL
      parameter (TOL = 1.d-18)
      real*8 a1, a2, b1, b2, x, y  ! coordinates of the vertices
      real*8 v1, v2, v1x, v1y, v2x, v2y 
      real*8 term1, term2, term3
      real*8 D, Dx, Dy ! D is the determinant (Cramer's rule)
c
      D = (a1 - x)*(b2 - y) - (b1 - x)*(a2 - y)
      if (dabs(D) .lt. TOL) then
         write(6,*)a1,a2,b1,b2,x,y
         call errormsg('Abort: 3 points of the triangle are collinear')
      endif
      term1 = 0.5*(((a1 - x)*(a1 + x) + (a2 - y)*(a2 + y))*(b2 - y)
     &        - ((b1 - x)*(b1 + x) + (b2 - y)*(b2 + y))*(a2 - y))
      v1 = term1/D
      term2 = 0.5*(((b1 - x)*(b1 + x) + (b2 - y)*(b2 + y))*(a1 - x)
     &        - ((a1 - x)*(a1 + x) + (a2 - y)*(a2 + y))*(b1 - x))
      v2 = term2/D
c
      Dx = a2 - b2
      Dy = b1 - a1
      term3 = 0.5d0*((b1 + a1)*(b1 - a1) + (b2 + a2)*(b2 - a2))
      v1x = (x - v1)*Dx/D
      v1y = (term3 + y*Dx - v1*Dy)/D
      v2x = (-term3 + x*Dy - v2*Dx)/D
      v2y = (y - v2)*Dy/D
c
      return
      end
c
c********************************************************************
c     subroutine consistency(xi1,xi2,n,coord,phi)
c     Purpose
c     =======
c     Check constant and linear consistency
c
c********************************************************************
c
      subroutine consistency(xi1,xi2,n,coord,phi)
      implicit none
c
      real*8 xi1, xi2, coord(2,*)
      real*8 phi(3,*)
      integer n

      integer i
      real*8 tol, sum, sumx, sumy, sumphix, sumphiy, sumderx, sumdery
      real*8 sumder_x, sumder_y
c
      tol = 1.0e-12
      sum = 0.d0
      sumx = 0.d0
      sumy = 0.d0
      sumphix = 0.d0
      sumphiy = 0.d0
      sumderx = 0.d0
      sumdery = 0.d0
      sumder_x = 0.d0
      sumder_y = 0.d0
      do 10 i = 1,n
         sum = sum + phi(1,i)
         sumx = sumx + phi(1,i)*coord(1,i)
         sumy = sumy + phi(1,i)*coord(2,i)
         sumphix = sumphix + phi(2,i)
         sumphiy = sumphiy + phi(3,i)
         sumderx = sumderx + phi(2,i)*coord(1,i)
         sumdery = sumdery + phi(3,i)*coord(2,i)
         sumder_x = sumder_x + phi(3,i)*coord(1,i)
         sumder_y = sumder_y + phi(2,i)*coord(2,i)
10    continue
c
      if ((dabs(1.d0-sum) .gt. tol) .or. (dabs(xi1-sumx) .gt. tol) 
     &    .or. (dabs(xi2-sumy) .gt. tol)
     &    .or. (dabs(sumphix) .gt. tol)
     &    .or. (dabs(sumphiy) .gt. tol)
     &    .or. (dabs(sumdery-1.d0) .gt. tol)
     &    .or. (dabs(sumdery-1.d0) .gt. tol)
     &    .or. (dabs(sumder_x) .gt. tol)
     &    .or. (dabs(sumder_y) .gt. tol)) then
         write(6,*)'WARNING: Consistency error at',xi1,xi2
         write(6,*)sum,sumx-xi1,sumy-xi2
         write(6,*)sumphix,sumphiy
         write(6,*)sumderx-1.d0,sumdery-1.d0,sumder_x,sumder_y
         write(6,*)
         call errormsg('Aborting . . ')
      endif
c
      return
      end
c
c********************************************************************
c     subroutine centroid_polygon(n,point)
c     Purpose
c     =======
c     Obtain the centroid of the n-gon
c********************************************************************
c
      subroutine centroid_polygon(n,point)
      implicit none
c
      integer n, i
      real*8 point(2), coord(2,n)
c
      call polycoord(n,coord)
      point(1) = 0.d0
      point(2) = 0.d0
      do 10 i = 1,n
         point(1) = point(1) + coord(1,i)
         point(2) = point(2) + coord(2,i)
10    continue
      point(1) = point(1)/dble(n)
      point(2) = point(2)/dble(n)
c
      end
c
c********************************************************************
c     subroutine polycoord(n,coord)
c     Purpose
c     =======
c     Obtain the nodal coordinates for an n-gon
c********************************************************************
c
      subroutine polycoord(n,coord)
      implicit none
c
      integer n, i
      real*8 coord(2,n), pi, theta, angle
c
      if (n .lt. 3) then
        call errormsg('Aborting: No of vertices must be >= 3')
      endif

      pi = acos(-1.d0)
      theta = 2.0*pi/dble(n)
      if (n .eq. 4) then ! four-node bi-unit square (FEM)
         coord(1,1) = -1.d0
         coord(2,1) = -1.d0
         coord(1,2) = 1.d0
         coord(2,2) = -1.d0
         coord(1,3) = 1.d0
         coord(2,3) = 1.d0
         coord(1,4) = -1.d0
         coord(2,4) = 1.d0
      else 
        do i = 1,n
c     nodes lie on a unit circle
            angle = dble(i-1)*theta 
            coord(1,i) = cos(angle)
            coord(2,i) = sin(angle)
        enddo
      endif
c
      end
c 
c***********************************************************************
c     logical function in_polygon(n, xy, px, py)
c     Finds if a point P(px,py) is inside or outside a polygon. 
c     Algorithm is based on ray tracing (even and odd crossings)
c***********************************************************************
c
      logical function in_polygon(n, xy, px, py)
      implicit none

      integer n
      real*8 xy(2,*), x(n), y(n)
      real*8 px, py
c
c     integer counter
c
      integer j, k
c
c     n = no of vertices/nodes of the polygon
c     x() and y() are n-dimensional arrays for the x- and y-coordinate of
c     the vertices in counter clockwise order
c

      do k = 1,n
        x(k) = xy(1,k)
        y(k) = xy(2,k)
      enddo

      in_polygon = .false.
      j = n
      do k = 1,n
         if ((((y(k) .le. py) .and. (py .lt. y(j))) .or.
     *       ((y(j) .le. py) .and. (py .lt. y(k)))) .and.
     *       (px .lt.
     *        (x(j) - x(k))*(py - y(k))/(y(j) - y(k)) + x(k))) then
           in_polygon = .not. in_polygon
         endif
         j = k
      enddo

      return
      end
c
c***********************************************************************
      subroutine errormsg(str)
c***********************************************************************
      character str*(*)
      write(6,*)str
      stop
      end
