/***********************************************************
*                                                          
*  Function for reading input data for the Natural Element
*  Method (NEM). The code is written in ANSI C. The input
*  data (file "input.dat") for the implementation of NEM 
*  consists of the following data blocks:
*
*  [1] Material Properties and Problem Type
*
*      Constant block data consisting of material properties
*      and problem type (plane stress or plane strain)
*
*      Constants
*          E = 3.0e7
*          nu = 0.3
*          Problem = 0 or 1 (0 -> plane stress; 1 -> plane strain)
*      &
*
*  [2] Nodal Grid
*
*      A six/seven-column data consisting of nodal coordinates,
*      nodal fixities, prescribed essential boundary conditions
*      (fix.i = 0 if u is prescribed at the node; otherwise 
*      fix.i = 1 if the node is free), and the nodal boundary
*      marker [optional] to indicate boundary nodes. The nodal 
*      block data is shown below:
*
*      Nodes
*         node.x   node.y   fix.x   fix.y  ux   uy  [node_marker]
*      &
*
*  [3] Delaunay Triangulation 
*
*      A four/seven-column data consisting of the Delaunay
*      triangulation of the nodes (cell connectivities), the
*      quadrature for numerical (Gauss) integration within each
*      triangle, and the neighbors [optional] for each triangle
*      (-1 for a neighbor indicates that the triangle is on the 
*      boundary) . The four/seven-column Delaunay triangulation 
*      block is shown below:
*
*      Delaunay
*         node1  node2   node3  quad  [neigh_t1 neigh_t2 neigh_t3]
*      &
*
*  [4] Traction Boundary Conditions
*
*      Each line consists of prescribed nodal tractions on nodes
*      belonging to a traction BC cell. The traction block data
*      is shown below:
*
*      Tractions
*         nbc_node1  tx1   ty1  nbc_node2  tx2   ty2
*      &
*
*            
*  External Functions: (1) constant_data(.)
*                      (2) nodal_data(.) 
*                      (3) triangle_data(.)
*                      (4) traction_data(.)
*  Function Prototypes   : File "prototypes.h"
*  Parameter Definitions : File "defines.h"
*
*  Author: N. Sukumar, E-mail: n-sukumar@nwu.edu
*  Organization: Theoretical & Applied Mechanics
*                Northwestern University
*                Evanston IL 60208
*  Date: April, 1997
*
***********************************************************/

#include <stdio.h>

#include "defines.h"
#include "prototypes.h"

void input_data(double constants[],int *no_of_nodes,double coord[][NSD],int fixity[][NSD],double uebc[][NSD],int node_boundary[],int *no_of_triangles,int tri_connect[][NODES_PER_TRIANGLE],int tri_quad[],int tri_boundary[],int *no_of_nbc_cells,int node_nbc[],double tx[],double ty[])

{
   int i, j;
   char tag_begin[200], input_file[] =  "input.dat";
   FILE *file_ptr;

   if ((file_ptr = fopen(input_file,"r")) == NULL) {
      fprintf(stderr,"Cannot open input file %s\n",input_file);
      exit(1);
   }

   *no_of_nbc_cells = 0; /* initialized to zero */
   while (fgets(tag_begin,200,file_ptr) != NULL) {
      switch (tag_begin[0]) {
         case '!':
		  break; /* comment line */
         case '&':
		  break; /* end of tag */
         case 'C':
		  constant_data(file_ptr,constants);
		  break;
         case 'N':
                  *no_of_nodes = nodal_data(file_ptr,coord,fixity,uebc,node_boundary);
		  break;
         case 'D':
                  *no_of_triangles = triangle_data(file_ptr,tri_connect,tri_quad,tri_boundary);
		  break;
         case 'T':
                  *no_of_nbc_cells = traction_data(file_ptr,node_nbc,tx,ty);
		  break;
         default:
		  printf("Error: %s is not a valid block tag\n",tag_begin);
		  exit(1);
		  break;
      }
   }

   fclose(file_ptr);

} /* end of input_data function */
